/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/***
 * \file: DM8127_Wlan.c
 *
 * \brief : source file for the DM8127_CSK WLAN.
 * 
 * \b \Descr: This is a diagonstic for the WLAN Module.
 * \n         The WLAN module is connected to the SDIO 0 line.
 * \n         The  DM8127_WLAN_EN is connected to the GP0 27th pin.
 * \n         The  DM8127_WLAN_IRQ is connected to the GP0 8th pin.
 */

#include "stdio.h"
#include "stdlib.h"
#include "string.h"
#include "DM8127_types.h"
#include "Dm8127_Platform.h"
#include "sdio.h"
#include "Wlan.h"
#include "gpio.h"

/** Vendor id of the WLAN CHIP */
#define SDIO_VENDOR_ID_TI			(0x0097)
/** Device ID of the WLAN Chip */
#define SDIO_DEVICE_ID_TI_WL1271	(0x4076)

/** Combined chip ID and the vendor ID */
#define TIWL1271_CHIPID				(SDIO_DEVICE_ID_TI_WL1271 << 16 | SDIO_VENDOR_ID_TI)
#define GPIO0_BASE_ADD			(0x48032000)
/* bit selection*/
#define BIT_SEL_GPIO (27u)

/***
 * \brief: GPIO enable function for the WLAN
 *
 * \b \descr: This function initialzes the GPIO bank 0.
 *
 * \param:  VOID
 * \return: VOID
 */
void GPIO_0_init
(
	void
)
{

	set_gpio_bank0 (BIT_SEL_GPIO, 1);
	delay(190000);
	set_gpio_bank0 (BIT_SEL_GPIO, 0);
	delay(190000);
	set_gpio_bank0 (BIT_SEL_GPIO, 1);
	delay(90000); /* settle down */

}


/***
 * \brief: WLAN test handler function.
 *
 * \b \descr: This function initialize the SDIO and start detection for
 *         WLAN card on DM8127 board.
 *
 * \param: VOID
 * \return: SUCCESS on successful detection the WLAN card on the board
 * \n       FAILED  failed to detct the WLAN card on the connected board.
 */
STATUS WLAN_Test
(
	void *testargs
)
{
	SINT32  cardtype;
	SINT32  is_highcap_card ;
	SINT32  retval = FAILURE;
	UINT32 chipid;
	STATUS s32status;

	cardtype = DM8127_INIT_VAL;
	is_highcap_card = DM8127_INIT_VAL;

	s32status = FAILURE;
	retval = FAILURE;
	chipid = DM8127_INIT_VAL;
#if 0
	s32status = WLAN_Init();
#endif
#ifdef DEBUG
		printf ("SDIO Tests:\n"
	        "*) Test communicates with an SDIO WLAN Chip and displays its ChipID\n"
	        "*) Ensure that the Boot Switches are selected as specified in the README\n");
#else
		 Uart_stringSend("\r\n==================================="
		 				 "\r\n            WLAN Test              "
						 "\r\n===================================\r\n\n");
    	Uart_stringSend ("Test communicates with an SDIO WLAN Chip and displays its ChipID\n\r");
#endif
    	GPIO_0_init();
    	SDIO_DEV_init((SDIO_controller_t *) HSWlan_BASE0);
    	cardtype = detect_card((SDIO_controller_t *) HSWlan_BASE0, &is_highcap_card);
	
    	if (cardtype == SDIO_CARD)
    	{
#ifdef DEBUG
    		printf ("\nSDIO Card Detected\n");
#else
    		Uart_stringSend("\r\nSDIO Card Detected\r\n");
#endif

    		retval = get_sdio_chipid((SDIO_controller_t *) HSWlan_BASE0, &chipid);
    		if (retval == SUCCESS)
    		{
    			if (chipid == TIWL1271_CHIPID)
    			{
#ifdef DEBUG
    				printf ("\tWL1271 WLAN Chip Detected with an Manfacturer ID of 0x%08x\n", chipid);
#else
    				platform_write("WL1271 WLAN Chip Detected with an Manfacturer ID of 0x%08x\n\n\r", chipid);
#endif
    			}
    			else
    			{
#ifdef DEBUG
    				printf ("\tAn SDIO Device detected with an Manfacturer ID of 0x%08x\n", chipid);
#else
    				platform_write ("An SDIO Device detected with an Manfacturer ID of 0x%08x\n\r", chipid);
#endif
    			}
    		}
    		else
    		{
#ifdef DEBUG
			printf ("\tUnable to read the chip id of the SDIO Card\n");
#else
			Uart_stringSend("Unable to read the chip id of the SDIO Card\n\r");
#endif
    		}
    	}
    	else
    	{
#ifdef DEBUG
    		printf ("\tUnable to detect an SDIO card\n");
#else
    		Uart_stringSend ("\nUnable to detect an SDIO card\n\r");
#endif

    		retval = FAILURE;
	}

	if (retval == SUCCESS)
	{
#ifdef DEBUG
		printf ("\nWLAN Test Passed\n");
#else
    		Uart_stringSend ("WLAN Test Passed\n\r");
#endif
    		s32status = SUCCESS;
	}

	else
	{
#ifdef DEBUG
		printf ("\nWLAN Test Failed\n");
#else
    		Uart_stringSend ("WLAN Test Failed\n\r");
#endif
	}
	Uart_stringSend("\r\n-------------------------X-----------------------------");
return(s32status);
}
